/*
 * Copyright (c) 2000, 2024, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package javax.management.relation;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamField;
import java.io.Serializable;

import javax.management.MBeanServer;

import javax.management.NotCompliantMBeanException;

/**
 * A RoleInfo object summarises a role in a relation type.
 *
 * <p>The <b>serialVersionUID</b> of this class is {@code 2504952983494636987L}.
 *
 * @since 1.5
 */
public class RoleInfo implements Serializable {

    private static final long serialVersionUID = 2504952983494636987L;
    /**
     * @serialField name String Role name
     * @serialField isReadable boolean Read access mode: {@code true} if role is readable
     * @serialField isWritable boolean Write access mode: {@code true} if role is writable
     * @serialField description String Role description
     * @serialField minDegree int Minimum degree (i.e. minimum number of referenced MBeans in corresponding role)
     * @serialField maxDegree int Maximum degree (i.e. maximum number of referenced MBeans in corresponding role)
     * @serialField referencedMBeanClassName String Name of class of MBean(s) expected to be referenced in corresponding role
     */
    private static final ObjectStreamField[] serialPersistentFields =
    {
      new ObjectStreamField("name", String.class),
      new ObjectStreamField("isReadable", boolean.class),
      new ObjectStreamField("isWritable", boolean.class),
      new ObjectStreamField("description", String.class),
      new ObjectStreamField("minDegree", int.class),
      new ObjectStreamField("maxDegree", int.class),
      new ObjectStreamField("referencedMBeanClassName", String.class)
    };

    //
    // Public constants
    //

    /**
     * To specify an unlimited cardinality.
     */
    public static final int ROLE_CARDINALITY_INFINITY = -1;

    //
    // Private members
    //

    /**
     * @serial Role name
     */
    private String name = null;

    /**
     * @serial Read access mode: {@code true} if role is readable
     */
    private boolean isReadable;

    /**
     * @serial Write access mode: {@code true} if role is writable
     */
    private boolean isWritable;

    /**
     * @serial Role description
     */
    private String description = null;

    /**
     * @serial Minimum degree (i.e. minimum number of referenced MBeans in corresponding role)
     */
    private int minDegree;

    /**
     * @serial Maximum degree (i.e. maximum number of referenced MBeans in corresponding role)
     */
    private int maxDegree;

    /**
     * @serial Name of class of MBean(s) expected to be referenced in corresponding role
     */
    private String referencedMBeanClassName = null;

    //
    // Constructors
    //

    /**
     * Constructor.
     *
     * @param roleName  name of the role.
     * @param mbeanClassName  name of the class of MBean(s) expected to
     * be referenced in corresponding role.  If an MBean <em>M</em> is in
     * this role, then the MBean server must return true for
     * {@link MBeanServer#isInstanceOf isInstanceOf(M, mbeanClassName)}.
     * @param read  flag to indicate if the corresponding role
     * can be read
     * @param write  flag to indicate if the corresponding role
     * can be set
     * @param min  minimum degree for role, i.e. minimum number of
     * MBeans to provide in corresponding role
     * Must be less than or equal to {@code max}.
     * (ROLE_CARDINALITY_INFINITY for unlimited)
     * @param max  maximum degree for role, i.e. maximum number of
     * MBeans to provide in corresponding role
     * Must be greater than or equal to {@code min}
     * (ROLE_CARDINALITY_INFINITY for unlimited)
     * @param descr  description of the role (can be null)
     *
     * @exception IllegalArgumentException  if null parameter
     * @exception InvalidRoleInfoException  if the minimum degree is
     * greater than the maximum degree.
     * @exception ClassNotFoundException As of JMX 1.2, this exception
     * can no longer be thrown.  It is retained in the declaration of
     * this class for compatibility with existing code.
     * @exception NotCompliantMBeanException  if the class mbeanClassName
     * is not a MBean class.
     */
    public RoleInfo(String roleName,
                    String mbeanClassName,
                    boolean read,
                    boolean write,
                    int min,
                    int max,
                    String descr)
    throws IllegalArgumentException,
           InvalidRoleInfoException,
           ClassNotFoundException,
           NotCompliantMBeanException {

        init(roleName,
             mbeanClassName,
             read,
             write,
             min,
             max,
             descr);
        return;
    }

    /**
     * Constructor.
     *
     * @param roleName  name of the role
     * @param mbeanClassName  name of the class of MBean(s) expected to
     * be referenced in corresponding role.  If an MBean <em>M</em> is in
     * this role, then the MBean server must return true for
     * {@link MBeanServer#isInstanceOf isInstanceOf(M, mbeanClassName)}.
     * @param read  flag to indicate if the corresponding role
     * can be read
     * @param write  flag to indicate if the corresponding role
     * can be set
     *
     * <P>Minimum and maximum degrees defaulted to 1.
     * <P>Description of role defaulted to null.
     *
     * @exception IllegalArgumentException  if null parameter
     * @exception ClassNotFoundException As of JMX 1.2, this exception
     * can no longer be thrown.  It is retained in the declaration of
     * this class for compatibility with existing code.
     * @exception NotCompliantMBeanException As of JMX 1.2, this
     * exception can no longer be thrown.  It is retained in the
     * declaration of this class for compatibility with existing code.
     */
    public RoleInfo(String roleName,
                    String mbeanClassName,
                    boolean read,
                    boolean write)
    throws IllegalArgumentException,
           ClassNotFoundException,
           NotCompliantMBeanException {

        try {
            init(roleName,
                 mbeanClassName,
                 read,
                 write,
                 1,
                 1,
                 null);
        } catch (InvalidRoleInfoException exc) {
            // OK : Can never happen as the minimum
            //      degree equals the maximum degree.
        }

        return;
    }

    /**
     * Constructor.
     *
     * @param roleName  name of the role
     * @param mbeanClassName  name of the class of MBean(s) expected to
     * be referenced in corresponding role.  If an MBean <em>M</em> is in
     * this role, then the MBean server must return true for
     * {@link MBeanServer#isInstanceOf isInstanceOf(M, mbeanClassName)}.
     *
     * <P>IsReadable and IsWritable defaulted to true.
     * <P>Minimum and maximum degrees defaulted to 1.
     * <P>Description of role defaulted to null.
     *
     * @exception IllegalArgumentException  if null parameter
     * @exception ClassNotFoundException As of JMX 1.2, this exception
     * can no longer be thrown.  It is retained in the declaration of
     * this class for compatibility with existing code.
     * @exception NotCompliantMBeanException As of JMX 1.2, this
     * exception can no longer be thrown.  It is retained in the
     * declaration of this class for compatibility with existing code.
      */
    public RoleInfo(String roleName,
                    String mbeanClassName)
    throws IllegalArgumentException,
           ClassNotFoundException,
           NotCompliantMBeanException {

        try {
            init(roleName,
                 mbeanClassName,
                 true,
                 true,
                 1,
                 1,
                 null);
        } catch (InvalidRoleInfoException exc) {
            // OK : Can never happen as the minimum
            //      degree equals the maximum degree.
        }

        return;
    }

    /**
     * Copy constructor.
     *
     * @param roleInfo the {@code RoleInfo} instance to be copied.
     *
     * @exception IllegalArgumentException  if null parameter
     */
    public RoleInfo(RoleInfo roleInfo)
        throws IllegalArgumentException {

        if (roleInfo == null) {
            // Revisit [cebro] Localize message
            String excMsg = "Invalid parameter.";
            throw new IllegalArgumentException(excMsg);
        }

        try {
            init(roleInfo.getName(),
                 roleInfo.getRefMBeanClassName(),
                 roleInfo.isReadable(),
                 roleInfo.isWritable(),
                 roleInfo.getMinDegree(),
                 roleInfo.getMaxDegree(),
                 roleInfo.getDescription());
        } catch (InvalidRoleInfoException exc3) {
            // OK : Can never happen as the minimum degree and the maximum
            //      degree were already checked at the time the roleInfo
            //      instance was created.
        }
    }

    //
    // Accessors
    //

    /**
     * Returns the name of the role.
     *
     * @return the name of the role.
     */
    public String getName() {
        return name;
    }

    /**
     * Returns read access mode for the role (true if it is readable).
     *
     * @return true if the role is readable.
     */
    public boolean isReadable() {
        return isReadable;
    }

    /**
     * Returns write access mode for the role (true if it is writable).
     *
     * @return true if the role is writable.
     */
    public boolean isWritable() {
        return isWritable;
    }

    /**
     * Returns description text for the role.
     *
     * @return the description of the role.
     */
    public String getDescription() {
        return description;
    }

    /**
     * Returns minimum degree for corresponding role reference.
     *
     * @return the minimum degree.
     */
    public int getMinDegree() {
        return minDegree;
    }

    /**
     * Returns maximum degree for corresponding role reference.
     *
     * @return the maximum degree.
     */
    public int getMaxDegree() {
        return maxDegree;
    }

    /**
     * <p>Returns name of type of MBean expected to be referenced in
     * corresponding role.</p>
     *
     * @return the name of the referenced type.
     */
    public String getRefMBeanClassName() {
        return referencedMBeanClassName;
    }

    /**
     * Returns true if the {@code value} parameter is greater than or equal to
     * the expected minimum degree, false otherwise.
     *
     * @param value  the value to be checked
     *
     * @return true if greater than or equal to minimum degree, false otherwise.
     */
    public boolean checkMinDegree(int value) {
        if (value >= ROLE_CARDINALITY_INFINITY &&
            (minDegree == ROLE_CARDINALITY_INFINITY
             || value >= minDegree)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Returns true if the {@code value} parameter is lower than or equal to
     * the expected maximum degree, false otherwise.
     *
     * @param value  the value to be checked
     *
     * @return true if lower than or equal to maximum degree, false otherwise.
     */
    public boolean checkMaxDegree(int value) {
        if (value >= ROLE_CARDINALITY_INFINITY &&
            (maxDegree == ROLE_CARDINALITY_INFINITY ||
             (value != ROLE_CARDINALITY_INFINITY &&
              value <= maxDegree))) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Returns a string describing the role info.
     *
     * @return a description of the role info.
     */
    public String toString() {
        StringBuilder result = new StringBuilder();
        result.append("role info name: " + name);
        result.append("; isReadable: " + isReadable);
        result.append("; isWritable: " + isWritable);
        result.append("; description: " + description);
        result.append("; minimum degree: " + minDegree);
        result.append("; maximum degree: " + maxDegree);
        result.append("; MBean class: " + referencedMBeanClassName);
        return result.toString();
    }

    //
    // Misc
    //

    // Initialization
    private void init(String roleName,
                      String mbeanClassName,
                      boolean read,
                      boolean write,
                      int min,
                      int max,
                      String descr)
            throws IllegalArgumentException,
                   InvalidRoleInfoException {

        if (roleName == null ||
            mbeanClassName == null) {
            // Revisit [cebro] Localize message
            String excMsg = "Invalid parameter.";
            throw new IllegalArgumentException(excMsg);
        }

        name = roleName;
        isReadable = read;
        isWritable = write;
        if (descr != null) {
            description = descr;
        }

        boolean invalidRoleInfoFlg = false;
        StringBuilder excMsgStrB = new StringBuilder();
        if (max != ROLE_CARDINALITY_INFINITY &&
            (min == ROLE_CARDINALITY_INFINITY ||
             min > max)) {
            // Revisit [cebro] Localize message
            excMsgStrB.append("Minimum degree ");
            excMsgStrB.append(min);
            excMsgStrB.append(" is greater than maximum degree ");
            excMsgStrB.append(max);
            invalidRoleInfoFlg = true;

        } else if (min < ROLE_CARDINALITY_INFINITY ||
                   max < ROLE_CARDINALITY_INFINITY) {
            // Revisit [cebro] Localize message
            excMsgStrB.append("Minimum or maximum degree has an illegal value, must be [0, ROLE_CARDINALITY_INFINITY].");
            invalidRoleInfoFlg = true;
        }
        if (invalidRoleInfoFlg) {
            throw new InvalidRoleInfoException(excMsgStrB.toString());
        }
        minDegree = min;
        maxDegree = max;

        referencedMBeanClassName = mbeanClassName;

        return;
    }

    /**
     * Deserializes a {@link RoleInfo} from an {@link ObjectInputStream}.
     */
    private void readObject(ObjectInputStream in)
            throws IOException, ClassNotFoundException {
      in.defaultReadObject();
    }


    /**
     * Serializes a {@link RoleInfo} to an {@link ObjectOutputStream}.
     */
    private void writeObject(ObjectOutputStream out)
            throws IOException {
      out.defaultWriteObject();
    }

}
