/*
 * Copyright (c) 2015, 2023, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package java.lang.invoke;

import jdk.internal.util.Preconditions;
import jdk.internal.vm.annotation.ForceInline;

import java.lang.invoke.VarHandle.VarHandleDesc;
import java.util.Objects;
import java.util.Optional;

import static java.lang.invoke.MethodHandleStatics.UNSAFE;

// -- This file was mechanically generated: Do not edit! -- //

final class VarHandleFloats {

    static sealed class FieldInstanceReadOnly extends VarHandle {
        final long fieldOffset;
        final Class<?> receiverType;

        FieldInstanceReadOnly(Class<?> receiverType, long fieldOffset) {
            this(receiverType, fieldOffset, FieldInstanceReadOnly.FORM, false);
        }

        protected FieldInstanceReadOnly(Class<?> receiverType, long fieldOffset,
                                        VarForm form, boolean exact) {
            super(form, exact);
            this.fieldOffset = fieldOffset;
            this.receiverType = receiverType;
        }

        @Override
        public FieldInstanceReadOnly withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadOnly(receiverType, fieldOffset, vform, true);
        }

        @Override
        public FieldInstanceReadOnly withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadOnly(receiverType, fieldOffset, vform, false);
        }

        @Override
        final MethodType accessModeTypeUncached(AccessType at) {
            return at.accessModeType(receiverType, float.class);
        }

        @Override
        public Optional<VarHandleDesc> describeConstable() {
            var receiverTypeRef = receiverType.describeConstable();
            var fieldTypeRef = float.class.describeConstable();
            if (!receiverTypeRef.isPresent() || !fieldTypeRef.isPresent())
                return Optional.empty();

            // Reflect on this VarHandle to extract the field name
            String name = VarHandles.getFieldFromReceiverAndOffset(
                receiverType, fieldOffset, float.class).getName();
            return Optional.of(VarHandleDesc.ofField(receiverTypeRef.get(), name, fieldTypeRef.get()));
        }

        @ForceInline
        static float get(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            return UNSAFE.getFloat(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset);
        }

        @ForceInline
        static float getVolatile(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            return UNSAFE.getFloatVolatile(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset);
        }

        @ForceInline
        static float getOpaque(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            return UNSAFE.getFloatOpaque(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset);
        }

        @ForceInline
        static float getAcquire(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            return UNSAFE.getFloatAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset);
        }

        static final VarForm FORM = new VarForm(FieldInstanceReadOnly.class, Object.class, float.class);
    }

    static final class FieldInstanceReadWrite extends FieldInstanceReadOnly {

        FieldInstanceReadWrite(Class<?> receiverType, long fieldOffset) {
            this(receiverType, fieldOffset, false);
        }

        private FieldInstanceReadWrite(Class<?> receiverType, long fieldOffset,
                                       boolean exact) {
            super(receiverType, fieldOffset, FieldInstanceReadWrite.FORM, exact);
        }

        @Override
        public FieldInstanceReadWrite withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadWrite(receiverType, fieldOffset, true);
        }

        @Override
        public FieldInstanceReadWrite withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadWrite(receiverType, fieldOffset, false);
        }

        @ForceInline
        static void set(VarHandle ob, Object holder, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putFloat(Objects.requireNonNull(handle.receiverType.cast(holder)),
                             handle.fieldOffset,
                             value);
        }

        @ForceInline
        static void setVolatile(VarHandle ob, Object holder, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putFloatVolatile(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                     handle.fieldOffset,
                                     value);
        }

        @ForceInline
        static void setOpaque(VarHandle ob, Object holder, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putFloatOpaque(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                   handle.fieldOffset,
                                   value);
        }

        @ForceInline
        static void setRelease(VarHandle ob, Object holder, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putFloatRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                    handle.fieldOffset,
                                    value);
        }

        @ForceInline
        static boolean compareAndSet(VarHandle ob, Object holder, float expected, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndSetFloat(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static float compareAndExchange(VarHandle ob, Object holder, float expected, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndExchangeFloat(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static float compareAndExchangeAcquire(VarHandle ob, Object holder, float expected, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndExchangeFloatAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static float compareAndExchangeRelease(VarHandle ob, Object holder, float expected, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndExchangeFloatRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetPlain(VarHandle ob, Object holder, float expected, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetFloatPlain(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSet(VarHandle ob, Object holder, float expected, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetFloat(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetAcquire(VarHandle ob, Object holder, float expected, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetFloatAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetRelease(VarHandle ob, Object holder, float expected, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetFloatRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static float getAndSet(VarHandle ob, Object holder, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndSetFloat(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static float getAndSetAcquire(VarHandle ob, Object holder, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndSetFloatAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static float getAndSetRelease(VarHandle ob, Object holder, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndSetFloatRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static float getAndAdd(VarHandle ob, Object holder, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndAddFloat(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                       handle.fieldOffset,
                                       value);
        }

        @ForceInline
        static float getAndAddAcquire(VarHandle ob, Object holder, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndAddFloatAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                       handle.fieldOffset,
                                       value);
        }

        @ForceInline
        static float getAndAddRelease(VarHandle ob, Object holder, float value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndAddFloatRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                       handle.fieldOffset,
                                       value);
        }


        static final VarForm FORM = new VarForm(FieldInstanceReadWrite.class, Object.class, float.class);
    }


    static sealed class FieldStaticReadOnly extends VarHandle {
        final Class<?> declaringClass;
        final Object base;
        final long fieldOffset;

        FieldStaticReadOnly(Class<?> declaringClass, Object base, long fieldOffset) {
            this(declaringClass, base, fieldOffset, FieldStaticReadOnly.FORM, false);
        }

        protected FieldStaticReadOnly(Class<?> declaringClass, Object base, long fieldOffset,
                                      VarForm form, boolean exact) {
            super(form, exact);
            this.declaringClass = declaringClass;
            this.base = base;
            this.fieldOffset = fieldOffset;
        }

        @Override
        public FieldStaticReadOnly withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadOnly(declaringClass, base, fieldOffset, vform, true);
        }

        @Override
        public FieldStaticReadOnly withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadOnly(declaringClass, base, fieldOffset, vform, false);
        }

        @Override
        public Optional<VarHandleDesc> describeConstable() {
            var fieldTypeRef = float.class.describeConstable();
            if (!fieldTypeRef.isPresent())
                return Optional.empty();

            // Reflect on this VarHandle to extract the field name
            var staticField = VarHandles.getStaticFieldFromBaseAndOffset(
                declaringClass, fieldOffset, float.class);
            var declaringTypeRef = declaringClass.describeConstable();
            if (!declaringTypeRef.isPresent())
                return Optional.empty();
            return Optional.of(VarHandleDesc.ofStaticField(declaringTypeRef.get(), staticField.getName(), fieldTypeRef.get()));
        }

        @Override
        final MethodType accessModeTypeUncached(AccessType at) {
            return at.accessModeType(null, float.class);
        }

        @ForceInline
        static float get(VarHandle ob) {
            FieldStaticReadOnly handle = (FieldStaticReadOnly) ob.target();
            return UNSAFE.getFloat(handle.base,
                                 handle.fieldOffset);
        }

        @ForceInline
        static float getVolatile(VarHandle ob) {
            FieldStaticReadOnly handle = (FieldStaticReadOnly) ob.target();
            return UNSAFE.getFloatVolatile(handle.base,
                                 handle.fieldOffset);
        }

        @ForceInline
        static float getOpaque(VarHandle ob) {
            FieldStaticReadOnly handle = (FieldStaticReadOnly) ob.target();
            return UNSAFE.getFloatOpaque(handle.base,
                                 handle.fieldOffset);
        }

        @ForceInline
        static float getAcquire(VarHandle ob) {
            FieldStaticReadOnly handle = (FieldStaticReadOnly) ob.target();
            return UNSAFE.getFloatAcquire(handle.base,
                                 handle.fieldOffset);
        }

        static final VarForm FORM = new VarForm(FieldStaticReadOnly.class, null, float.class);
    }

    static final class FieldStaticReadWrite extends FieldStaticReadOnly {

        FieldStaticReadWrite(Class<?> declaringClass, Object base, long fieldOffset) {
            this(declaringClass, base, fieldOffset, false);
        }

        private FieldStaticReadWrite(Class<?> declaringClass, Object base, long fieldOffset,
                                     boolean exact) {
            super(declaringClass, base, fieldOffset, FieldStaticReadWrite.FORM, exact);
        }

        @Override
        public FieldStaticReadWrite withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadWrite(declaringClass, base, fieldOffset, true);
        }

        @Override
        public FieldStaticReadWrite withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadWrite(declaringClass, base, fieldOffset, false);
        }

        @ForceInline
        static void set(VarHandle ob, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            UNSAFE.putFloat(handle.base,
                             handle.fieldOffset,
                             value);
        }

        @ForceInline
        static void setVolatile(VarHandle ob, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            UNSAFE.putFloatVolatile(handle.base,
                                     handle.fieldOffset,
                                     value);
        }

        @ForceInline
        static void setOpaque(VarHandle ob, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            UNSAFE.putFloatOpaque(handle.base,
                                   handle.fieldOffset,
                                   value);
        }

        @ForceInline
        static void setRelease(VarHandle ob, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            UNSAFE.putFloatRelease(handle.base,
                                    handle.fieldOffset,
                                    value);
        }

        @ForceInline
        static boolean compareAndSet(VarHandle ob, float expected, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.compareAndSetFloat(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }


        @ForceInline
        static float compareAndExchange(VarHandle ob, float expected, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.compareAndExchangeFloat(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static float compareAndExchangeAcquire(VarHandle ob, float expected, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.compareAndExchangeFloatAcquire(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static float compareAndExchangeRelease(VarHandle ob, float expected, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.compareAndExchangeFloatRelease(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetPlain(VarHandle ob, float expected, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.weakCompareAndSetFloatPlain(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSet(VarHandle ob, float expected, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.weakCompareAndSetFloat(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetAcquire(VarHandle ob, float expected, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.weakCompareAndSetFloatAcquire(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetRelease(VarHandle ob, float expected, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.weakCompareAndSetFloatRelease(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static float getAndSet(VarHandle ob, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndSetFloat(handle.base,
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static float getAndSetAcquire(VarHandle ob, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndSetFloatAcquire(handle.base,
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static float getAndSetRelease(VarHandle ob, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndSetFloatRelease(handle.base,
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static float getAndAdd(VarHandle ob, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndAddFloat(handle.base,
                                       handle.fieldOffset,
                                       value);
        }

        @ForceInline
        static float getAndAddAcquire(VarHandle ob, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndAddFloatAcquire(handle.base,
                                       handle.fieldOffset,
                                       value);
        }

        @ForceInline
        static float getAndAddRelease(VarHandle ob, float value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndAddFloatRelease(handle.base,
                                       handle.fieldOffset,
                                       value);
        }

        static final VarForm FORM = new VarForm(FieldStaticReadWrite.class, null, float.class);
    }


    static final class Array extends VarHandle {
        final int abase;
        final int ashift;

        Array(int abase, int ashift) {
            this(abase, ashift, false);
        }

        private Array(int abase, int ashift, boolean exact) {
            super(Array.FORM, exact);
            this.abase = abase;
            this.ashift = ashift;
        }

        @Override
        public Array withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new Array(abase, ashift, true);
        }

        @Override
        public Array withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new Array(abase, ashift, false);
        }

        @Override
        public Optional<VarHandleDesc> describeConstable() {
            var arrayTypeRef = float[].class.describeConstable();
            if (!arrayTypeRef.isPresent())
                return Optional.empty();

            return Optional.of(VarHandleDesc.ofArray(arrayTypeRef.get()));
        }

        @Override
        final MethodType accessModeTypeUncached(AccessType at) {
            return at.accessModeType(float[].class, float.class, int.class);
        }


        @ForceInline
        static float get(VarHandle ob, Object oarray, int index) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return array[index];
        }

        @ForceInline
        static void set(VarHandle ob, Object oarray, int index, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            array[index] = value;
        }

        @ForceInline
        static float getVolatile(VarHandle ob, Object oarray, int index) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.getFloatVolatile(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase);
        }

        @ForceInline
        static void setVolatile(VarHandle ob, Object oarray, int index, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            UNSAFE.putFloatVolatile(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    value);
        }

        @ForceInline
        static float getOpaque(VarHandle ob, Object oarray, int index) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.getFloatOpaque(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase);
        }

        @ForceInline
        static void setOpaque(VarHandle ob, Object oarray, int index, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            UNSAFE.putFloatOpaque(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    value);
        }

        @ForceInline
        static float getAcquire(VarHandle ob, Object oarray, int index) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.getFloatAcquire(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase);
        }

        @ForceInline
        static void setRelease(VarHandle ob, Object oarray, int index, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            UNSAFE.putFloatRelease(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    value);
        }

        @ForceInline
        static boolean compareAndSet(VarHandle ob, Object oarray, int index, float expected, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.compareAndSetFloat(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    expected,
                    value);
        }

        @ForceInline
        static float compareAndExchange(VarHandle ob, Object oarray, int index, float expected, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.compareAndExchangeFloat(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    expected,
                    value);
        }

        @ForceInline
        static float compareAndExchangeAcquire(VarHandle ob, Object oarray, int index, float expected, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.compareAndExchangeFloatAcquire(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    expected,
                    value);
        }

        @ForceInline
        static float compareAndExchangeRelease(VarHandle ob, Object oarray, int index, float expected, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.compareAndExchangeFloatRelease(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    expected,
                    value);
        }

        @ForceInline
        static boolean weakCompareAndSetPlain(VarHandle ob, Object oarray, int index, float expected, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.weakCompareAndSetFloatPlain(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    expected,
                    value);
        }

        @ForceInline
        static boolean weakCompareAndSet(VarHandle ob, Object oarray, int index, float expected, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.weakCompareAndSetFloat(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    expected,
                    value);
        }

        @ForceInline
        static boolean weakCompareAndSetAcquire(VarHandle ob, Object oarray, int index, float expected, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.weakCompareAndSetFloatAcquire(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    expected,
                    value);
        }

        @ForceInline
        static boolean weakCompareAndSetRelease(VarHandle ob, Object oarray, int index, float expected, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.weakCompareAndSetFloatRelease(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    expected,
                    value);
        }

        @ForceInline
        static float getAndSet(VarHandle ob, Object oarray, int index, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.getAndSetFloat(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    value);
        }

        @ForceInline
        static float getAndSetAcquire(VarHandle ob, Object oarray, int index, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.getAndSetFloatAcquire(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    value);
        }

        @ForceInline
        static float getAndSetRelease(VarHandle ob, Object oarray, int index, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.getAndSetFloatRelease(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    value);
        }

        @ForceInline
        static float getAndAdd(VarHandle ob, Object oarray, int index, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.getAndAddFloat(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    value);
        }

        @ForceInline
        static float getAndAddAcquire(VarHandle ob, Object oarray, int index, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.getAndAddFloatAcquire(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    value);
        }

        @ForceInline
        static float getAndAddRelease(VarHandle ob, Object oarray, int index, float value) {
            Array handle = (Array)ob;
            float[] array = (float[]) oarray;
            return UNSAFE.getAndAddFloatRelease(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << handle.ashift) + handle.abase,
                    value);
        }

        static final VarForm FORM = new VarForm(Array.class, float[].class, float.class, int.class);
    }
}
